<?php
/**
 * Meta Options Helper Functions
 * This file is loaded at 'after_setup_theme' hook with 4 priority.
 *
 * @package    Magazine News Byte
 * @subpackage Library Premium
 */

/**
 * Helper function to return the theme meta option value.
 * If no value has been saved, it returns $default.
 * If no $default provided, it checks the default meta options array.
 *
 * Similar to 'hoot_get_mod()'
 * 
 * @since 3.0.0
 * @access public
 * @param string $name Use empty string to return entire option values array.
 * @param int $post_id (Optional - if used within the loop)
 * @param mixed $default
 * @param string $meta_key The meta key from which to get values. Themes should use default 'mainbox'
 *                         as ID for the main meta options box id for brevity.
 *                         (in include/admin/meta-options.php)
 * @return mixed
 */
function hoot_get_metaoption( $name = '', $post_id = 0, $default = NULL, $meta_key = 'mainbox' ) {

	/*** Return meta option value if set ***/

	// cache
	static $options = NULL;

	// If no post id, get current post's ID (in the loop)
	$post_id = intval( $post_id );
	if ( empty( $post_id ) )
		$post_id = get_the_ID();

	// Find and Return value
	if ( !empty( $post_id ) ) {

		// Populate cache if not available
		$options[ $post_id ] = empty( $options[ $post_id ] ) ? array() : $options[ $post_id ];
		if ( !isset( $options[ $post_id ][ $meta_key ] ) ) {
			// Use post id instead of global posts else loops like WC Shop will return 'product'
			$post_type = get_post_type( $post_id );

			$hoot_metaoptions = Hoot_Metaoptions::get_instance();
			$metabox_settings = $hoot_metaoptions->get_options( $post_type, $meta_key );

			$fields = !empty( $metabox_settings['options'] ) ? $metabox_settings['options'] : array();
			$values = get_post_meta( $post_id, '_hoot_meta_' . $meta_key , true );
			$values = apply_filters( 'hoot_get_metaoption_value', $values, $post_id, $post_type, $meta_key );

			$options[ $post_id ][ $meta_key ] = hoot_sanitize_get_meta( $values, $fields );
		}

		// Return all meta options if requested
		if ( empty( $name ) )
			return $options[ $post_id ][ $meta_key ];

		// Return single meta option if available
		if ( isset( $options[ $post_id ][ $meta_key ][ $name ] ) ) {
			// Add exceptions: If a value has been set but is empty, this gives the option to return default values in such cases. Simply return $override as (bool)true.
			$override = apply_filters( 'hoot_get_metaoption_empty_values', false, $options[ $post_id ][ $meta_key ][ $name ] );
			if ( $override !== true )
				return $options[ $post_id ][ $meta_key ][ $name ];
		}

	}

	/*** Return default if provided ***/

	if ( $default !== NULL )
		return $default;

	/*** Return default theme meta value ***/

	// Use post id instead of global posts else loops like WC Shop will return 'product'
	$post_type = get_post_type( $post_id );

	// Cache
	static $defaults = NULL;

	// Populate cache if not available
	$defaults[ $post_type ] = empty( $defaults[ $post_type ] ) ? array() : $defaults[ $post_type ];
	if ( !isset( $defaults[ $post_type ][ $meta_key ] ) ) {
		$hoot_metaoptions = Hoot_Metaoptions::get_instance();
		$metabox_settings = $hoot_metaoptions->get_options( $post_type, $meta_key );

		$fields = !empty( $metabox_settings['options'] ) ? $metabox_settings['options'] : array();
		$defaults = array();
		foreach ( $fields as $opid => $opfield ) {
			if ( isset( $opfield['std'] ) )
				$defaults[ $opid ] = $opfield['std'];
		}

		$defaults[ $post_type ][ $meta_key ] = hoot_sanitize_get_meta( $defaults, $fields );
	}

	// Return all meta options if requested
	// This condition will never be reached in 99.99% cases i.e. $post_id will most likely be available (if it isn't then this function is not being used properly)
	// Adding for brevity (or modify above code if this is needed in future)
	if ( empty( $name ) )
		return $defaults[ $post_type ][ $meta_key ];

	// Return single meta option if available
	if ( isset( $defaults[ $post_type ][ $meta_key ][ $name ] ) )
		return $defaults[ $post_type ][ $meta_key ][ $name ];

	/*** We dont have anything! ***/
	return false;
}

/**
 * Sanitization function for saving meta options values to database
 * Also used to sanitize return value of 'hoot_get_metaoption()'
 *
 * @since 3.0.0
 * @param array $values
 * @param array $fields
 * @return array
 */
function hoot_sanitize_get_meta( $values, $fields ){
	if ( empty( $values ) || empty( $fields ) || !is_array( $values ) || !is_array( $fields ) )
		return array();

	$output = array();

	foreach ( $fields as $id => $field ) {

		/* Skip if the field does not have an id/type */
		if ( !is_string( $id ) || !isset( $field['type'] ) )
			continue;

		/* Skip if instance value is not set (except for checkbox) - This covers heading, subheading, html, info etc */
		if ( !isset( $values[ $id ] ) && $field['type'] != 'checkbox' )
			continue;

		/* Skip if specifically told to skip */
		if ( isset( $field['settings'] ) && !empty( $field['settings']['skipsave'] ) )
			continue;

		if ( !isset( $field['sanitize'] ) ) {

			/* Default field sanitization */
			switch ( $field['type'] ) {
				case 'text':
					$output[ $id ] = hoot_sanitize_text( $values[ $id ] );
					break;
				case 'textarea':
					$output[ $id ] = hoot_sanitize_textarea( $values[ $id ] );
					break;
				case 'checkbox':
					$values[ $id ] = ( !empty( $values[ $id ] ) ) ? $values[ $id ] : '';
					$output[ $id ] = hoot_sanitize_checkbox( $values[ $id ] );
					break;
				case 'select':
				case 'radio':
				case 'images':
					$output[ $id ] = ( isset( $field['options'] ) ) ? hoot_sanitize_enum( $values[ $id ], $field['options'] ) : '';
					break;
				case 'image':
					$output[ $id ] = absint( $values[ $id ] ); // Image is stored as ID
					break;
				case 'color':
					$output[ $id ] = sanitize_hex_color( $values[ $id ] );
					break;
				case 'upload':
					$output[ $id ] = esc_url( $values[ $id ] );
					break;
			}

		} elseif ( isset( $field['sanitize'] ) ) {

			/* Custom sanitizations for specific field. Example, a text input has a url */
			switch( $field['sanitize'] ) {
				case 'url':
					$output[ $id ] = esc_url_raw( $values[ $id ] );
					break;
				case 'integer':
					if ( $values[ $id ] === '0' || $values[ $id ] === 0 ) {
						$output[ $id ] = 0;
					} else {
						$output[ $id ] = intval( $values[ $id ] );
						$output[ $id ] = ( !empty( $output[ $id ] ) ) ? $output[ $id ] : '';
					}
					break;
				case 'absint':
					if ( $values[ $id ] === '0' || $values[ $id ] === 0 ) {
						$output[ $id ] = 0;
					} else {
						$output[ $id ] = absint( $values[ $id ] );
						$output[ $id ] = ( !empty( $output[ $id ] ) ) ? $output[ $id ] : '';
					}
					break;
				case 'email':
					$output[ $id ] = ( is_email( $values[ $id ] ) ) ? sanitize_email( $values[ $id ] ) : '';
					break;
				// Allow custom sanitization functions
				default:
					$output[ $id ] = apply_filters( 'hoot_sanitize_get_meta_field', $values[ $id ], $field['sanitize'], $values );
			}

		}

	}
	return $output;
}