<?php
/**
 * Meta Options Interface Build.
 * This file is loaded at 'after_setup_theme' hook with 4 priority.
 *
 * @package    Magazine News Byte
 * @subpackage Library Premium
 */

/**
 * Class wrapper with methods for building the metaoptions interface
 * 
 * @since 3.0.0
 */
final class Hoot_Metaoptions_Interface {

	/**
	 * Holds the instance of this class.
	 *
	 * @since 3.0.0
	 * @access private
	 * @var object
	 */
	private static $instance;

	/**
	 * The array for storing namespace prefix.
	 *
	 * @since 3.0.0
	 * @access public
	 * @var string
	 */
	public $prefix = '';

	/**
	 * The array for allowed tags in sanitization.
	 *
	 * @since 3.0.0
	 * @access public
	 * @var string
	 */
	public $allowedtags = '';

	/**
	 * Protected constructor to prevent creating a new instance of the
	 * Singleton from outside of this class.
	 *
	 * @since 3.0.0
	 */
	protected function __construct() {
		$this->prefix = hoot_data()->basetemplate_slug;

		global $allowedtags;
		$this->allowedtags = $allowedtags;
		$this->allowedtags['br'] = array();
		$this->allowedtags['a']['target'] = true;
		$this->allowedtags['img']['src'] = array();
	}

	/**
	 * Generates the options fields that are used in the form.
	 *
	 * @since 3.0.0
	 * @param array   $options  Options array
	 * @param array   $values   Options values
	 * @param string  $prefix   Options namespace
	 */
	function build( $options = array(), $values = array(), $prefix = '' ) {

		$this->prefix = ( $prefix ) ? $prefix : hoot_data()->basetemplate_slug;

		if ( empty( $options ) || ( !empty( $values ) && !is_array( $values ) ) )
			return;

		foreach ( $options as $id => $field ) {

			// Settings Keys:
			// {*} skipsave, containerclass, condition/match
			// {textarea} code, rows, placeholder
			// {text} size, placeholder
			$field = wp_parse_args( (array) $field, array( 	'name'     => '',
															'desc'     => '',
															'type'     => '',
															'settings' => array(),
															'std'      => '',
															'options'  => array(),
															'fields'   => array(),
															'sanitize' => '',
													) );
			if ( !is_string( $id ) || empty( $field['type'] ) ) continue;

			$value = false;
			if ( isset( $values[ $id ] ) ) { $value = $values[ $id ];
				// Strip slashes of non-array value which aren't code
				if ( ! is_array( $value ) && 
					 ! ( $field['type'] == 'textarea' && !empty( $field['settings'] ) && !empty( $field['settings']['code'] ) )
					) {
					$value = stripslashes( $value );
				}
			} elseif ( !empty( $field['std'] ) ) $value = $field['std'];

			// Print the field HTML
			$this->render_field( $id, $field, $value );

		}

	}

	/**
	 * Generates the options field
	 *
	 * @since 3.0.0
	 * @param string $id       Option ID
	 * @param array  $field    Option array
	 * @param array  $value    Option value
	 * @param array  $repeater Array of ['groupid'] for prefixing to field names
	 */
	function render_field( $id, $field, $value, $repeater = array() ) {

		extract( $field, EXTR_SKIP );
		$boxclass = ( isset( $settings['containerclass'] ) ) ? ' ' . hoot_sanitize_html_classes( $settings['containerclass'] ) : '';
		if ( isset( $settings['condition'] ) && isset( $settings['match'] ) ) {
			$boxclass .= ' show-on-select';
			$data_args = 'data-selector="' . esc_attr( $settings['condition'] ) . '" data-selectorval="' . esc_attr( $settings['match'] ) . '"';
		} else $data_args = '';

		if ( $type == 'heading' ) {
			echo '<div class="hoot-metaoption-header' . $boxclass . '" ' . $data_args . '><p>' . esc_html( $field['name'] ) . '</p></div>' ."\n";
		} elseif( $type == 'subheading' ) {
			echo '<div class="hoot-metaoption-subheader' . $boxclass . '" ' . $data_args . '><p>' . esc_html( $field['name'] ) . '</p></div>' ."\n";
		} elseif ( $type == 'html' ) {
			if ( isset( $field['std'] ) ) echo wp_kses_post( $field['std'] );
		} else {
			?>
			<div id="hoot-<?php echo sanitize_html_class( $id ) ?>" class="hoot-metaoption-box hoot-metaoption-<?php echo sanitize_html_class( $type ) . $boxclass ?>" <?php echo $data_args; ?>>
				<?php if ( !empty( $name ) ) : ?>
					<h4 class="hoot-metaoption-label"><?php echo esc_html( $field['name'] ) ?></h4>
				<?php endif; ?>
				<div class="hoot-metaoption-control">
				<div class="hoot-metaoption">
					<?php
					switch ( $type ):

						case 'info' :
							if ( !empty( $desc ) )
								echo wp_kses( $desc, $this->allowedtags );
							$desc = '';
							break;

						case 'text' :
							$size = ( isset( $settings['size'] ) && is_numeric( $settings['size'] ) ) ? ' size="' . $settings['size'] . '" ' : '';
							$placeholder = ( isset( $settings['placeholder'] ) ) ? ' placeholder="' . esc_attr( $settings['placeholder'] ) . '" ' : '';
							?><input type="text" name="<?php echo $this->hoot_get_field_name( $id, $repeater ) ?>" id="<?php echo $this->hoot_get_field_id( $id, $repeater ) ?>" value="<?php echo esc_attr( $value ) ?>" class="hoot-field-text" <?php echo $size . $placeholder; ?> /><?php
							break;

						case 'textarea' :
							$rows = ( isset( $settings['rows'] ) && is_numeric( $settings['rows'] ) ) ? intval( $settings['rows'] ) : 8;
							$placeholder = ( isset( $settings['placeholder'] ) ) ? ' placeholder="' . esc_attr( $settings['placeholder'] ) . '" ' : '';
							$codeclass = ( isset( $settings['code'] ) && true === $settings['code'] ) ? ' code' : '';
							$value = ( isset( $settings['code'] ) && true === $settings['code'] ) ? htmlspecialchars_decode( $value ) : $value;
							?><textarea name="<?php echo $this->hoot_get_field_name( $id, $repeater ) ?>" id="<?php echo $this->hoot_get_field_id( $id, $repeater ) ?>" class="hoot-field-textarea<?php echo $codeclass ?>" rows="<?php echo $rows; ?>" <?php echo $placeholder ?>><?php echo esc_textarea( $value ) ?></textarea><?php
							break;

						case 'checkbox':
							?><input type="checkbox" name="<?php echo $this->hoot_get_field_name( $id, $repeater ) ?>" id="<?php echo $this->hoot_get_field_id( $id, $repeater ) ?>" class="hoot-field-checkbox" <?php checked( !empty( $value ) ) ?> />
							<?php if ( !empty( $desc ) ) : ?>
								<label class="hoot-metaoption-desc" for="<?php echo $this->hoot_get_field_id( $id, $repeater ) ?>"><?php echo wp_kses( $desc, $this->allowedtags ) ?></label>
							<?php endif;
							$desc = '';
							break;

						case 'select':
							?><select name="<?php echo $this->hoot_get_field_name( $id, $repeater ) ?>" id="<?php echo $this->hoot_get_field_id( $id, $repeater ) ?>" class="hoot-field-select">
								<?php foreach( $options as $k => $v ) : ?>
									<option value="<?php echo esc_attr($k) ?>" <?php selected($k, $value) ?>><?php echo esc_html($v) ?></option>
								<?php endforeach; ?>
							</select><?php
							break;

						case 'radio': case 'images':
							?><ul id="<?php echo $this->hoot_get_field_id( $id, $repeater ) ?>-list" class="hoot-field-list hoot-field-list-<?php echo $type ?>">
								<?php foreach( $options as $k => $v ) : ?>
									<li class="hoot-field-list-item">
										<input type="radio" class="hoot-field-radio" name="<?php echo $this->hoot_get_field_name( $id, $repeater ) ?>" id="<?php echo $this->hoot_get_field_id( $id, $repeater ) . '-' . sanitize_html_class( $k ) ?>" value="<?php echo esc_attr($k) ?>" <?php checked( $k, $value ) ?>>
										<label for="<?php echo $this->hoot_get_field_id( $id, $repeater ) . '-' . sanitize_html_class( $k ) ?>"><?php echo ( 'radio' === $type ) ? $v : "<img class='hoot-field-image-picker-img' src='" . esc_url( $v ) . "'>" ?></label>
									</li>
								<?php endforeach; ?>
							</ul><?php
							break;

						case 'color':
							$default_color = ( isset( $field['std'] ) && $value !=  $field['std'] ) ? ' data-default-color="' . $field['std'] . '" ' : '';
							?><input type="text" name="<?php echo $this->hoot_get_field_name( $id, $repeater ) ?>" id="<?php echo $this->hoot_get_field_id( $id, $repeater ) ?>" value="<?php echo esc_attr( $value ) ?>" class="hoot-field-color" <?php echo $default_color; ?> /><?php
							break;

						case 'upload':
							$load = apply_filters( 'hoot_metaoptions_upload', false );
							if ( $load ) {
								$output = '';
								$_id = $this->hoot_get_field_id( $id, $repeater );
								$name = $this->hoot_get_field_name( $id, $repeater );
								$class = ( $value ) ? ' has-file' : '';

								$output .= '<input id="' . esc_attr( $_id ) . '" class="upload' . $class . '" type="text" name="' . esc_attr( $name ) . '" value="' . $value . '" placeholder="' . __('No file chosen', 'magazine-news-byte-premium') .'" />' . "\n";
								if ( function_exists( 'wp_enqueue_media' ) ) {
									if ( ( $value == '' ) ) {
										$output .= '<input id="upload-' . esc_attr( $_id ) . '" class="hoot-upload-button button" type="button" value="' . __( 'Upload', 'magazine-news-byte-premium' ) . '" />' . "\n";
									} else {
										$output .= '<input id="remove-' . esc_attr( $_id ) . '" class="hoot-remove-file button" type="button" value="' . __( 'Remove', 'magazine-news-byte-premium' ) . '" />' . "\n";
									}
								} else {
									$output .= '<p><i>' . __( 'Upgrade your version of WordPress for full media support.', 'magazine-news-byte-premium' ) . '</i></p>';
								}
								$output .= '<div class="screenshot" id="' . $_id . '-image" style="max-width: 100px; background: #f5f5f5; padding: 5px; text-align: center; margin-top: 5px; border: solid 1px #ddd;">' . "\n";
								if ( $value != '' ) {
									$remove = ''; //'<a class="hoot-remove-image">Remove</a>';
									$image = preg_match( '/(^.*\.jpg|jpeg|png|gif|ico*)/i', $value );
									if ( $image ) {
										$output .= '<img src="' . $value . '" alt="" />' . $remove;
									} else {
										$parts = explode( "/", $value );
										for( $i = 0; $i < sizeof( $parts ); ++$i ) {
											$title = $parts[$i];
										}
										// No output preview if it's not an image.
										$output .= '';
										// Standard generic output if it's not an image.
										$title = __( 'View File', 'magazine-news-byte-premium' );
										$output .= '<div class="no-image"><span class="file_link"><a href="' . $value . '" target="_blank" rel="external">'.$title.'</a></span></div>';
									}
								}
								$output .= '</div>' . "\n";
								echo $output;
							}
							break;

					endswitch;
					?>
				</div><!-- hoot-metaoption -->
				<?php if ( !empty( $desc ) ) echo '<div class="hoot-metaoption-desc">' . wp_kses( $desc, $this->allowedtags ) . '</div>'; ?>
			</div><!-- hoot-metaoption-control -->
			</div><!-- hoot-metaoption-box -->
			<?php
		}

	}

	/**
	 * @since 3.0.0
	 * @param $id
	 * @param array $repeater
	 * @return mixed|string
	 */
	public function hoot_get_field_name( $id, $repeater = array() ) {
		if ( empty( $repeater ) ) {
			return esc_attr( $this->prefix . '[' . $id . ']' );
		} else {
			return esc_attr( $this->prefix . '[' . $id . ']' );
		}
	}

	/**
	 * Get the ID of this field.
	 *
	 * @since 3.0.0
	 * @param $id
	 * @param array $repeater
	 * @return string
	 */
	public function hoot_get_field_id( $id, $repeater = array() ) {
		if ( empty( $repeater ) ) {
			return sanitize_html_class( $this->prefix . '-' . $id );
		} else {
			return sanitize_html_class( $this->prefix . '-' . $id );
		}
	}

	/**
	 * Returns the instance.
	 *
	 * @since 3.0.0
	 * @access public
	 * @return object
	 */
	public static function get_instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

}

/** Initialize **/
Hoot_Metaoptions_Interface::get_instance();